# Copyright (c) 2025, benilerouge.ddns.net
# Licensed under the MIT License.

import os
import subprocess
import shutil
from ffmpeg_utils import FFmpegOptimizer

SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
DEFAULT_TEMP_DIR = os.path.join(SCRIPT_DIR, "temp")
os.makedirs(DEFAULT_TEMP_DIR, exist_ok=True)

AUDIO_OFFSET_SEC = 0.0

class Section:
    def __init__(self, start, end, action, blend_v_section=None, blend_v_prev=None, blend_mode='before'):
        self.start = start
        self.end = end
        self.action = action
        self.blend_v_section = blend_v_section
        self.blend_v_prev = blend_v_prev
        self.blend_mode = blend_mode
        self.use_micro = False
        self.mic_path = None
        self.manual_times = None

    def __repr__(self):
        if self.action == "blend":
            if self.blend_mode == "manual" and self.manual_times:
                timestr = f"Manual {self.manual_times[0]:.2f}-{self.manual_times[1]:.2f}"
            else:
                timestr = f"{self.blend_v_section*100:.0f}%/{self.blend_v_prev*100:.0f}%"
            return f"Section({self.start:.2f}-{self.end:.2f}, blend {self.blend_mode} {timestr}{', mic' if self.use_micro else ''})"
        else:
            return f"Section({self.start:.2f}-{self.end:.2f}, {self.action})"

class TempFiles:
    def __init__(self):
        self.files = set()

    def add(self, path):
        self.files.add(path)

    def update(self, paths):
        for p in paths:
            self.add(p)

    def clean(self, keep=None):
        keep = set(keep or [])
        for f in list(self.files):
            if os.path.exists(f) and f not in keep:
                try:
                    os.remove(f)
                except Exception:
                    pass
            self.files.discard(f)

temp_files = TempFiles()

def extract_audio_wav(video_path, temp_dir=DEFAULT_TEMP_DIR):
    audio_temp = os.path.join(temp_dir, "audio_temp.wav")
    temp_files.add(audio_temp)
    hw_args = FFmpegOptimizer.get_hwaccel_args()
    subprocess.run([
        "ffmpeg", "-y",
        *hw_args,
        "-i", video_path,
        "-vn",
        "-ar", "44100",
        "-ac", "2",
        "-c:a", "pcm_s16le",
        audio_temp
    ], check=True)
    return audio_temp

def apply_audio_edits(video_path, duration, sections, temp_dir=DEFAULT_TEMP_DIR,
                      att_volume=0.2, fade_duration=1.0):
    audio_temp = extract_audio_wav(video_path, temp_dir)
    part_files = []
    current = 0.0
    sections_edit = sorted([s for s in sections if s.action in ('replace', 'blend')], key=lambda s: s.start)
    hw_args = FFmpegOptimizer.get_hwaccel_args()

    for idx, s in enumerate(sections_edit):
        start, end = s.start, s.end
        dur_sec = end - start
        if current < start:
            part_keep = os.path.join(temp_dir, f"part_keep_{idx}.wav")
            temp_files.add(part_keep)
            subprocess.run([
                "ffmpeg", "-y",
                *hw_args,
                "-i", audio_temp,
                "-ss", str(current),
                "-to", str(start),
                "-c", "copy",
                part_keep
            ], check=True)
            part_files.append(part_keep)

        if s.action == "replace":
            copy_start = start - dur_sec
            if copy_start < 0:
                raise ValueError(f"Section {idx+1}: Pas assez d'audio avant pour remplacer.")
            part_replace = os.path.join(temp_dir, f"part_replace_{idx}.wav")
            temp_files.add(part_replace)
            subprocess.run([
                "ffmpeg", "-y",
                *hw_args,
                "-i", audio_temp,
                "-ss", str(copy_start),
                "-to", str(start),
                "-c", "copy",
                part_replace
            ], check=True)
            part_files.append(part_replace)

        elif s.action == "blend":
            if s.blend_mode == "manual" and s.manual_times:
                copy_start, copy_end = s.manual_times
            elif s.blend_mode == "before":
                copy_start = start - dur_sec
                copy_end = start
            else:
                copy_start = end
                copy_end = end + dur_sec

            if copy_start < 0 or copy_end > duration:
                raise ValueError(f"Section {idx+1}: Pas assez d'audio pour le mixage.")

            v_section = s.blend_v_section if s.blend_v_section is not None else 0.5
            v_prev = s.blend_v_prev if s.blend_v_prev is not None else 1.0

            blend_low = os.path.join(temp_dir, f"blend_low_{idx}.wav")
            blend_source = os.path.join(temp_dir, f"blend_source_{idx}.wav")
            blend_mix = os.path.join(temp_dir, f"blend_mix_{idx}.wav")
            temp_files.update([blend_low, blend_source, blend_mix])

            # Extraction segment atténué
            subprocess.run([
                "ffmpeg", "-y",
                *hw_args,
                "-i", audio_temp,
                "-ss", str(start),
                "-to", str(end),
                "-af", f"volume={v_section}",
                blend_low
            ], check=True)

            # Source blend
            if getattr(s, "use_micro", False) and s.mic_path:
                subprocess.run([
                    "ffmpeg", "-y",
                    *hw_args,
                    "-i", s.mic_path,
                    "-ac", "2",
                    "-ar", "44100",
                    "-t", f"{dur_sec:.3f}",
                    "-af", "apad",
                    blend_source
                ], check=True)
            else:
                subprocess.run([
                    "ffmpeg", "-y",
                    *hw_args,
                    "-i", audio_temp,
                    "-ss", str(copy_start),
                    "-to", str(copy_end),
                    "-af", f"volume={v_prev}",
                    blend_source
                ], check=True)

            # Mixage
            subprocess.run([
                "ffmpeg", "-y",
                *hw_args,
                "-i", blend_low,
                "-i", blend_source,
                "-filter_complex", "amix=inputs=2:duration=first:dropout_transition=0:normalize=0",
                "-c:a", "pcm_s16le",
                blend_mix
            ], check=True)

            part_files.append(blend_mix)

        current = end

    if current < duration:
        final_keep = os.path.join(temp_dir, "part_keep_final.wav")
        temp_files.add(final_keep)
        subprocess.run([
            "ffmpeg", "-y",
            *hw_args,
            "-i", audio_temp,
            "-ss", str(current),
            "-to", str(duration),
            "-c", "copy",
            final_keep
        ], check=True)
        part_files.append(final_keep)

    if len(part_files) == 1:
        audio_result = part_files[0]
    else:
        concat_txt = os.path.join(temp_dir, "concat_list.txt")
        with open(concat_txt, "w") as f:
            for pf in part_files:
                f.write(f"file '{pf}'\n")
        temp_files.add(concat_txt)

        audio_result = os.path.join(temp_dir, "audio_concatenated.wav")
        temp_files.add(audio_result)

        subprocess.run([
            "ffmpeg", "-y",
            *hw_args,
            "-f", "concat",
            "-safe", "0",
            "-i", concat_txt,
            "-c", "copy",
            audio_result
        ], check=True)

    audio_final = os.path.join(temp_dir, "audio_final.wav")
    temp_files.add(audio_final)
    shutil.copy(audio_result, audio_final)

    final_name = os.path.splitext(video_path)[0] + "_modified.mp4"

    ffmpeg_cmd = [
        "ffmpeg", "-y",
        *hw_args,
        "-i", video_path,
        "-i", audio_final,
        "-map", "0:v:0",
        "-map", "1:a:0",
        "-c:v", "copy",
        "-c:a", "aac",
        "-movflags", "+faststart",
        "-shortest",
        final_name
    ]
    if AUDIO_OFFSET_SEC != 0.0:
        ffmpeg_cmd.insert(2, "-itsoffset")
        ffmpeg_cmd.insert(3, str(AUDIO_OFFSET_SEC))

    subprocess.run(ffmpeg_cmd, check=True)
    return final_name

def apply_video_cut(video_path, duration, sections, temp_dir=DEFAULT_TEMP_DIR):
    cuts = sorted([s for s in sections if s.action == 'cut'], key=lambda s: s.start)
    if not cuts:
        return video_path

    keep_ranges = []
    last_end = 0.0
    for s in cuts:
        if last_end < s.start:
            keep_ranges.append((last_end, s.start))
        last_end = s.end
    if last_end < duration:
        keep_ranges.append((last_end, duration))
    if not keep_ranges:
        raise Exception("No portion to keep")

    temp_files_list = []
    hw_args = FFmpegOptimizer.get_hwaccel_args()
    for i, (start, end) in enumerate(keep_ranges):
        temp_name = os.path.join(temp_dir, f"temp_keep_{i+1}.mp4")
        temp_files.add(temp_name)
        subprocess.run([
            "ffmpeg", "-y",
            *hw_args,
            "-ss", str(start),
            "-to", str(end),
            "-i", video_path,
            "-c:v", "copy",
            "-c:a", "copy",
            temp_name
        ], check=True)
        temp_files_list.append(temp_name)

    concat_file = os.path.join(temp_dir, "concat_keep.txt")
    with open(concat_file, "w") as f:
        for fn in temp_files_list:
            f.write(f"file '{fn}'\n")
    temp_files.add(concat_file)

    final_name = os.path.splitext(video_path)[0] + "_cut.mp4"
    subprocess.run([
        "ffmpeg", "-y",
        *hw_args,
        "-f", "concat",
        "-safe", "0",
        "-i", concat_file,
        "-c:v", "copy",
        "-c:a", "copy",
        "-movflags", "+faststart",
        final_name
    ], check=True)
    return final_name

def apply_all_edits_func(video_path, duration, sections, temp_dir=DEFAULT_TEMP_DIR):
    filename_base = os.path.splitext(video_path)[0]
    audio_edited = any(s.action in ('replace', 'fade', 'blend') for s in sections)
    video_cut = any(s.action == 'cut' for s in sections)
    current_path = video_path

    if audio_edited:
        current_path = apply_audio_edits(current_path, duration, sections, temp_dir=temp_dir)
    if video_cut:
        current_path = apply_video_cut(current_path, duration, sections, temp_dir=temp_dir)

    if (audio_edited or video_cut) and current_path != video_path:
        # Modification ici : on ne combine plus les suffixes
        if audio_edited and video_cut:
            final_name = filename_base + "_audiomodified_videocut.mp4"
        elif audio_edited:
            final_name = filename_base + "_audiomodified.mp4"
        elif video_cut:
            final_name = filename_base + "_videocut.mp4"

        if os.path.exists(final_name):
            os.remove(final_name)
        os.rename(current_path, final_name)
        current_path = final_name

    temp_files.clean(keep=[video_path, current_path])
    for suffix in ["_modified.mp4", "_cut.mp4"]:
        candidate = filename_base + suffix
        if os.path.exists(candidate) and candidate != current_path:
            try:
                os.remove(candidate)
            except Exception:
                pass

    return current_path
